//////////////////////////////////////////////////////////////
//
//         -= Using gradients with linear mapping =-
//
//
//  
// Skal99.   Pascal.Massimino@ens.fr.
// http://www.eleves.ens.fr:8080/home/massimin/
//////////////////////////////////////////////////////////////

#include <math.h>
#include <stdio.h>

#ifndef TRUE
#define TRUE (0==0)
#define FALSE (1==0)
#endif

#ifndef NULL
#define NULL  0
#endif

#ifndef M_PI
#define M_PI   3.1415926535
#endif

#include "video.h"
#include "matrix.h"

static unsigned char Texture[256*256];

typedef struct {

   VECTOR P;            // vertex
   float U, V;          // field associated

} VTX;

typedef struct { 

   float X, Y;

} POINT;

#define MAX_VTX 4

typedef struct {

   int Nb_Vtx;
   VTX Vtx[MAX_VTX];

   VECTOR dUw, dVw, dw;       // Original field gradients
   float  Area;

} POLY3D;

typedef struct {

  int      Nb_Vtx;
  POINT    Vtx[MAX_VTX];      // projected vertex
  VECTOR   TdUw, TdVw, Tdw;   // transformed gradients
  float    Area;

} POLY2D;

//////////////////////////////////////////////////////////////////////

#define NB_POLYS  6
#define Z_FRONT   10.0

//#define Z_FRONT   0.0       // uncomment, to test the m=0 bug...

POLY3D Poly3D[NB_POLYS] = {
 { 4, {
     {{-10.0, -10.0, Z_FRONT},  25.0,  0.0 },
     {{ 10.0, -10.0, Z_FRONT},  50.0,  0.0 },
     {{ 10.0,  10.0, Z_FRONT},  50.0, 25.0 },
     {{-10.0,  10.0, Z_FRONT},   0.0,  0.0 } } },
 { 4, {
     {{-10.0, -10.0,-10.0},     0.0,  0.0 },
     {{-10.0,  10.0,-10.0},     0.0, 25.0 },
     {{ 10.0,  10.0,-10.0},    25.0, 25.0 },
     {{ 10.0, -10.0,-10.0},     0.0,  0.0 } } },
 { 4, {
     {{-10.0, -10.0,-10.0},     25.0, 25.0 },
     {{-10.0, -10.0, Z_FRONT},  50.0, 25.0 },
     {{-10.0,  10.0, Z_FRONT},  50.0, 50.0 },
     {{-10.0,  10.0,-10.0},      0.0,  0.0 } } },
 { 4, {
     {{ 10.0, -10.0,-10.0},     75.0, 25.0 },
     {{ 10.0,  10.0,-10.0},    100.0, 50.0 },
     {{ 10.0,  10.0, Z_FRONT},  75.0, 50.0 },
     {{ 10.0, -10.0, Z_FRONT},   0.0,  0.0 } } },
 { 4, {
     {{-10.0,  10.0,-10.0},      0.0,   0.0 },
     {{-10.0,  10.0, Z_FRONT}, 128.0,   0.0 },
     {{ 10.0,  10.0, Z_FRONT}, 128.0, 128.0 },
     {{ 10.0,  10.0,-10.0},      0.0,   0.0 } } },
 { 4, {
     {{-10.0, -10.0,-10.0},     0.0,  75.0 },
     {{ 10.0, -10.0,-10.0},     0.0, 100.0 },
     {{ 10.0, -10.0, Z_FRONT}, 25.0, 100.0 },
     {{-10.0, -10.0, Z_FRONT},  0.0,   0.0 } } }
};

//////////////////////////////////////////////////////////////

// #define USE_WBUFFER

#define MAX_H  800

int Scan_Start = MAX_H; 
int Scan_End   = 0;
int Scan_Xi[MAX_H];     // left points
int Scan_Xf[MAX_H];     // right points

static void Draw_Scans( POLY2D *Poly2d )
{
   int y;
   unsigned char *Ptr;
   float Uwo, Vwo, wo;
   float DxUw, DxVw, Dxw;

   if ( Scan_Start>=Scan_End ) return;

   y = Scan_Start;
   Ptr = VID_MEM(y);

   DxUw = Poly2d->TdUw[0];
   DxVw = Poly2d->TdVw[0];
   Dxw  = Poly2d->Tdw[0];

      // Compute field values on starting point  (0,The_H/2-y)
      // We're using the fact that we can retreive the value
      // at point P of any field by using: F(P) = P.dF

   Screen_Vo[0] =-Xc;
   Screen_Vo[1] = Yc-y;       // beware of sign
   Screen_Vo[2] = Focal;
   wo  = Dot_Product( Screen_Vo, Poly2d->Tdw );
   Uwo = Dot_Product( Screen_Vo, Poly2d->TdUw );
   Vwo = Dot_Product( Screen_Vo, Poly2d->TdVw );
   
   DxVw *= 256.0;       // Vw-field is pre-scaled by 256.0
   Vwo *= 256.0;

         // Inner-loop time!! Well let's keep it as simple as possible
         // no fancy linearly interpolated Uw/w here...

   for( ; y<Scan_End; ++y )
   {
      float Uw, Vw, w, zo;
      int Xo, Count;
      unsigned char *Dst;

      Xo = Scan_Xi[y];
      Count = Xo - Scan_Xf[y];
      if ( Count>=0 ) goto Skip;

      Dst = Ptr + Scan_Xf[y];
      Uw = Uwo + Xo * DxUw;
      Vw = Vwo + Xo * DxVw;
      w  = wo  + Xo * Dxw;

#if defined( USE_WBUFFER )
      {
         int W, dW;
         W = (int)( w*64.0*65536.0 ); dW = (int)( Dxw*64.0*65536.0 );
         for( ; Count<0; Count++ )
         {
            Dst[Count] = (W>>16)&0xff; W += dW;
         }
      }
#else

      zo = 1.0/w;
      for( ; Count<0; Count++ )
      {
         int u, v;
         u = (int)( Uw*zo ); v = (int)( Vw*zo ); // wow! 16 ticks ;)
         Dst[Count] = Texture[ (u&0xff) | (v&0xff00) ];
         zo -= Dxw*zo*zo;
         Uw += DxUw; 
         Vw += DxVw;
      }
#endif

            // add some red egdes...

      Dst[-1] = 0xff;
      Ptr[Scan_Xi[y]] = 0xff;

Skip:
      Uwo -= Poly2d->TdUw[1];           // Beware of sign
      Vwo -= Poly2d->TdVw[1] * 256.0;
      wo  -= Poly2d->Tdw[1];
      Ptr += VID_PITCH;
   }
}

//////////////////////////////////////////////////////////////
//          Ultra-Simple Rasterizator II (The Return)
//////////////////////////////////////////////////////////////

static void Scan_Edge( POINT *Pi, POINT *Pf )
{
   int yi, yf, y, X;
   float Slope, x;

   yi = (int)ceil( Pi->Y );
   if ( yi>The_H ) yi = The_H;
   else if ( yi<0 ) yi = 0;

   yf = (int)ceil( Pf->Y );
   if ( yf>The_H ) yf = The_H;
   else if ( yf<0 ) yf = 0;

   if ( yi==yf ) return;

   Slope  = ( Pf->X-Pi->X ) / ( Pf->Y-Pi->Y );

   if ( yi<yf )
   {
      x = Pi->X + ( (float)yi - Pi->Y ) * Slope;
      for( y=yi; y<yf; ++y )
      {
         X = (int)floor(x);
         if ( X<0 ) X = 0;
         else if ( X>The_W ) X = The_W;
         Scan_Xf[y] = X;
         x += Slope;
      }
   }
   else
   {
      x = Pf->X + ( (float)yf - Pf->Y ) * Slope;
      for( y=yf; y<yi; ++y )
      {
         X = (int)floor(x);
         if ( X<0 ) X = 0;
         else if ( X>The_W ) X = The_W;
         Scan_Xi[y] = X;
         x += Slope;
      }
      y = yi; yi = yf; yf = y;
      
   }
   if ( yi<Scan_Start ) Scan_Start = yi;
   if ( yf>Scan_End ) Scan_End = yf;
}

static void Scan_Convert( POLY2D *Poly2d )
{
   int i; 
   for( i=0; i<Poly2d->Nb_Vtx-1; ++i )
      Scan_Edge( &Poly2d->Vtx[i], &Poly2d->Vtx[i+1] );
   Scan_Edge( &Poly2d->Vtx[i], &Poly2d->Vtx[0] );
}

//////////////////////////////////////////////////////////////

static void Draw_Background_Scans( POLY2D *Poly2d )
{
   int y;
   unsigned char *Ptr;
   float Uwo, Vwo, wo;
   float DxUw, DxVw, Dxw;

   Ptr = VID_MEM( 0 );

   DxUw = Poly2d->TdUw[0];
   DxVw = Poly2d->TdVw[0];
   Dxw  = Poly2d->Tdw[0];

         // upper-left point

   Screen_Vo[0] =-Xc;
   Screen_Vo[1] = Yc;
   Screen_Vo[2] = Focal;
   Uwo = Dot_Product( Screen_Vo, Poly2d->TdUw );
   Vwo = Dot_Product( Screen_Vo, Poly2d->TdVw );
   wo  = Dot_Product( Screen_Vo, Poly2d->Tdw );

   DxVw *= 256.0;       // Vw-field is pre-scaled by 256.0
   Vwo *= 256.0;

   for( y=0; y<The_H; ++y )
   {
      float Uw, Vw, w, zo;
      unsigned char *Dst;
      int Count;

      Dst = Ptr + The_W;
      Uw = Uwo; Vw = Vwo; w  = wo;
      zo = 1.0/w;

      for( Count=-The_W; Count<0; Count++ )
      {
         int u, v;
         u = (int)( Uw*zo ); v = (int)( Vw*zo );
         Dst[Count] = Texture[ (u&0x00ff) | (v&0xff00) ];
         zo -= Dxw*zo*zo;
         Uw += DxUw; 
         Vw += DxVw;
      }
      Uwo -= Poly2d->TdUw[1];           // Beware of sign
      Vwo -= Poly2d->TdVw[1] * 256.0;
      wo  -= Poly2d->Tdw[1];
      Ptr += VID_PITCH;
   }
}

//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////

static void Store_Original_Gradients( POLY3D *Poly3d )
{
   VECTOR Q0, Q1, Q2;
   VECTOR Df;
   float N2;

        // Store Qi = Pj^Pk

   Cross_Product( Q0, Poly3d->Vtx[1].P, Poly3d->Vtx[2].P );
   Cross_Product( Q1, Poly3d->Vtx[2].P, Poly3d->Vtx[0].P );
   Cross_Product( Q2, Poly3d->Vtx[0].P, Poly3d->Vtx[1].P );

        // the 'normal'

   Poly3d->dw[0]  = Q0[0] + Q1[0] + Q2[0];
   Poly3d->dw[1]  = Q0[1] + Q1[1] + Q2[1];
   Poly3d->dw[2]  = Q0[2] + Q1[2] + Q2[2];

        // Mixed product

   Poly3d->Area = Dot_Product( Poly3d->Vtx[0].P, Q0 );    

        // gradients: N^Df
        // with Df = (f1-f2).P0 + (f2-f0).P1 + (f0-f1).P2

        // we could throw away the original .U and .V field and replace them
        // by the gradients, but better not obfuscate the code too much...

   Scale_Vector( Df, Poly3d->Vtx[0].P, Poly3d->Vtx[1].U - Poly3d->Vtx[2].U );
   Add_Scaled_Vector_Eq( Df, Poly3d->Vtx[1].P, Poly3d->Vtx[2].U - Poly3d->Vtx[0].U );
   Add_Scaled_Vector_Eq( Df, Poly3d->Vtx[2].P, Poly3d->Vtx[0].U - Poly3d->Vtx[1].U );
   Cross_Product( Poly3d->dUw, Poly3d->dw, Df );

   Scale_Vector( Df, Poly3d->Vtx[0].P, Poly3d->Vtx[1].V - Poly3d->Vtx[2].V );
   Add_Scaled_Vector_Eq( Df, Poly3d->Vtx[1].P, Poly3d->Vtx[2].V - Poly3d->Vtx[0].V );
   Add_Scaled_Vector_Eq( Df, Poly3d->Vtx[2].P, Poly3d->Vtx[0].V - Poly3d->Vtx[1].V );
   Cross_Product( Poly3d->dVw, Poly3d->dw, Df );

   N2 = 1.0f/Norm_Squared( Poly3d->dw );
   Scale_Vector_Eq( Poly3d->dUw, N2 );
   Scale_Vector_Eq( Poly3d->dVw, N2 );

      // If you want to have a real-normalized normal
      // stored in dw, just uncomment these two lines

//   Scale_Vector_Eq( Poly3d->dw, sqrt(N2) );
//   Poly3d->Area *= sqrt(N2);

}

static int Transform_Poly3D( POLY2D *Poly2d, POLY3D *Poly3d, MATRIX2 M )
{
   int i;
   float m, c;

      // 'm' is the new Area in homogeneous coordinates space,
      // After projection, you can retreive the actual surface 
      // on screen by multiplying by w0.w1.w2  (see text).

      // note: &M[21] stores the vector  '-itM.T'. Hence the minus sign...

   m = Poly3d->Area - Dot_Product( &M[21], Poly3d->dw );

   if ( m>=0.0 ) return( -1 );     // Backface cull

   Poly2d->Area = m;

   m = 1.0f / m;        // <= this is the only divide required
                        //    for computing gradients...

      //  Apply transform rules for gradients:
      //   dw' = 1/m' * itM. dw
      //   dF' = itM.dF - ((itM.T).dF)*dw'

   Scale_Vector( Poly2d->Tdw, Poly3d->dw, m );
   nA_V_Eq( Poly2d->Tdw, M );

   nA_V( Poly2d->TdUw, M, Poly3d->dUw );
   c = Dot_Product( &M[21], Poly3d->dUw );
   Add_Scaled_Vector_Eq( Poly2d->TdUw, Poly2d->Tdw, c );

   nA_V( Poly2d->TdVw, M, Poly3d->dVw );
   c = Dot_Product( &M[21], Poly3d->dVw );
   Add_Scaled_Vector_Eq( Poly2d->TdVw, Poly2d->Tdw, c );


         // Transform and project now

   for( i=0; i<Poly3d->Nb_Vtx; ++i )
   {
      float w;
      VECTOR Tmp;

      A_V( Tmp, M, Poly3d->Vtx[i].P );    // transform vertex

      w = Tmp[2];
      if ( w<1.0e-6 ) return(-1);  // simple pseudo z-clip

      w = Focal / w;
      Poly2d->Vtx[i].X = Xc + Tmp[0]*w;
      Poly2d->Vtx[i].Y = Yc - Tmp[1]*w;
   }
   Poly2d->Nb_Vtx = Poly3d->Nb_Vtx;

   return( 0 );
}

//////////////////////////////////////////////////////////////

static void Option_Error( char *S )
{
   fprintf( stderr, "Missing value after '%s' option\n" );
   exit(1);
}
static void Missing( char *S )
{
   fprintf( stderr, "can't scan value after '%s' option\n" );
   exit(1);
}
static void Help( )
{
   printf( " Usage: grad [options] [3ds_file]\n" );
   printf( "    Options:\n" );
   printf( "-h ................... this help\n" );
   printf( "-bck ................. draws only background\n" );
#if defined(__LINUX__)
   printf( "-size <int> <int> .... Size Width and Height\n" );
#endif
   printf( " Keys: a,d,w,s,+,-,1,2,q  + mouse\n" );
   exit(1);
}

//////////////////////////////////////////////////////////////

static void Draw_Polys( )
{
   int n;
   POLY2D Poly2D;

   Clear_Video( );

   for( n=0; n<NB_POLYS; ++n )
   {
      Scan_Start = The_H;
      Scan_End = 0;

      if ( Transform_Poly3D( &Poly2D, &Poly3D[n], Mo )==-1 )
         continue;

      Scan_Convert( &Poly2D );
      Draw_Scans( &Poly2D );
   }
}

static void Draw_Background( )
{
   POLY2D Poly2D;

   if ( Transform_Poly3D( &Poly2D, &Poly3D[0], Mo ) )
      return;
   Draw_Background_Scans( &Poly2D );
}

//////////////////////////////////////////////////////////////

int main( int argc, char **argv )
{
   int i;
   int Do_Move, Do_Scale, Do_Background;

   Do_Background = FALSE;
   Do_Move = TRUE;
   Do_Scale = FALSE;

   for( i=1; i<argc; ++i )
   {
      if ( !strcmp( argv[i], "-h" ) ) Help( );
#if defined(__LINUX__)
      else if ( !strcmp( argv[i], "-size" ) ) { 
         if (++i==argc ) Option_Error( argv[i-1] );
         if ( sscanf( argv[i], "%d", &The_W ) != 1 ) Missing( argv[i-1] );
         if ( The_W<20 ) The_W = 20;
         if (++i==argc ) Option_Error( argv[i-2] );
         if ( sscanf( argv[i], "%d", &The_H ) != 1 ) Missing( argv[i-2] );
         if ( The_H<20 ) The_H = 20;
         if ( The_H>MAX_H ) The_H = MAX_H;
      }
#endif
      else if ( !strcmp( argv[i], "-bck" ) ) Do_Background = TRUE;
   }

         // Initial view direction and position 

   Init_Positions( );

      // Init all gradients once for all

   for( i=0; i<NB_POLYS; ++i )
      Store_Original_Gradients( &Poly3D[i] );

      // let's set up some strange texture

   for( i=0; i<256*256; ++i ) 
      Texture[i] = ((i^(i>>8))*16+i-(i>>8))&0xff;


   Init_Video( );

   while( 1 )
   {
      int x, y;
      switch( Get_Key( ) )
      {
         default: case -1: break;
         case 'q': goto Finish; break;
         case '1': Do_Move = 1-Do_Move; break;
         case '2': Do_Scale = 1-Do_Scale; break;
         case 'd': Position[0] += 1.0; break;
         case 'a': Position[0] -= 1.0; break;
         case 'w': Position[1] -= 1.0; break;
         case 's': Position[1] += 1.0; break;
         case '+': Position[2] -= 3.0; break;
         case '-': Position[2] += 3.0; break;
      }

      Get_Mouse_Coord( &x, &y );
      Yaw = M_PI*(x-Xc)/The_W *0.35;
      Pitch = M_PI*(Yc-y)/The_H *0.45;

      Setup_Transform( Do_Move, Do_Scale );

      if ( Do_Background ) 
      {
         Draw_Background( );
         Position[1] += 1.0;  // makes the camera fly over the rotating plane
      }
      else Draw_Polys( );

      Update_Video( );
   }

Finish:   
   Close_Video( );
   return( 0 );
}

//////////////////////////////////////////////////////////////
